/**
 * XML import/export functionality for Cite Linker Pro
 */

/**
 * Export patterns to XML format
 */
window.exportToXML = function(patterns, settings) {
  const xmlDoc = document.implementation.createDocument(null, 'cite-linker-config');
  const root = xmlDoc.documentElement;
  
  // Add version info
  root.setAttribute('version', '1.0');
  root.setAttribute('exported', new Date().toISOString());
  
  // Add settings section
  const settingsEl = xmlDoc.createElement('settings');
  
  const settingsToExport = {
    enabled: settings.enabled || true,
    linkStyle: settings.linkStyle || 'dashed',
    showTooltips: settings.showTooltips !== false,
    domainMode: settings.domainMode || 'all',
    domains: settings.domains || []
  };
  
  Object.entries(settingsToExport).forEach(([key, value]) => {
    const settingEl = xmlDoc.createElement('setting');
    settingEl.setAttribute('name', key);
    
    if (Array.isArray(value)) {
      value.forEach(item => {
        const itemEl = xmlDoc.createElement('item');
        itemEl.textContent = item;
        settingEl.appendChild(itemEl);
      });
    } else {
      settingEl.textContent = value.toString();
    }
    
    settingsEl.appendChild(settingEl);
  });
  
  root.appendChild(settingsEl);
  
  // Add patterns section
  const patternsEl = xmlDoc.createElement('patterns');
  
  patterns.forEach(pattern => {
    const patternEl = xmlDoc.createElement('pattern');
    
    // Add pattern attributes
    patternEl.setAttribute('id', pattern.id || '');
    patternEl.setAttribute('enabled', pattern.enabled !== false);
    patternEl.setAttribute('predefined', pattern.isPredefined || false);
    
    // Add pattern elements
    const nameEl = xmlDoc.createElement('name');
    nameEl.textContent = pattern.name || '';
    patternEl.appendChild(nameEl);
    
    const regexEl = xmlDoc.createElement('regex');
    regexEl.textContent = pattern.regex || '';
    patternEl.appendChild(regexEl);
    
    const baseUrlEl = xmlDoc.createElement('baseUrl');
    baseUrlEl.textContent = pattern.baseUrl || '';
    patternEl.appendChild(baseUrlEl);
    
    if (pattern.description) {
      const descEl = xmlDoc.createElement('description');
      descEl.textContent = pattern.description;
      patternEl.appendChild(descEl);
    }
    
    patternsEl.appendChild(patternEl);
  });
  
  root.appendChild(patternsEl);
  
  // Serialize to string
  const serializer = new XMLSerializer();
  const xmlString = serializer.serializeToString(xmlDoc);
  
  // Format XML for readability
  return formatXML(xmlString);
};

/**
 * Import patterns from RegexContainer XML format
 */
window.importFromRegexContainerXML = function(xmlString) {
  try {
    // First, extract comments from raw XML text
    const commentMap = new Map();
    const regexContainerMatches = xmlString.matchAll(/<RegexContainer>\s*<!--\s*([^>]+?)\s*-->/g);
    let containerIndex = 0;
    for (const match of regexContainerMatches) {
      commentMap.set(containerIndex, match[1].trim());
      containerIndex++;
    }
    
    const parser = new DOMParser();
    const xmlDoc = parser.parseFromString(xmlString, 'text/xml');
    
    // Check for parsing errors
    const parserError = xmlDoc.querySelector('parsererror');
    if (parserError) {
      throw new Error('Invalid XML format: ' + parserError.textContent);
    }
    
    const root = xmlDoc.documentElement;
    
    if (root.tagName !== 'ArrayOfRegexContainer') {
      throw new Error('Invalid XML format: root element should be ArrayOfRegexContainer');
    }
    
    const result = {
      settings: {},
      patterns: [],
      metadata: {
        format: 'RegexContainer',
        imported: new Date().toISOString()
      }
    };
    
    // Parse RegexContainer elements
    const regexContainers = root.querySelectorAll('RegexContainer');
    regexContainers.forEach((container, index) => {
      const expressionEl = container.querySelector('Expression');
      const hyperlinkBaseEl = container.querySelector('HyperlinkBase');
      
      if (expressionEl && hyperlinkBaseEl) {
        // Get comment from our extracted map
        const comment = commentMap.get(index);
        let name = comment || `Imported Pattern ${index + 1}`;
        let description = comment || '';
        
        const pattern = {
          id: Date.now().toString() + '_' + index,
          name: name,
          regex: expressionEl.textContent.trim(),
          baseUrl: hyperlinkBaseEl.textContent.trim(),
          description: description,
          enabled: true,
          isPredefined: false
        };
        
        result.patterns.push(pattern);
      }
    });
    
    return {
      success: true,
      data: result
    };
    
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
};

/**
 * Import patterns from XML format (original cite-linker format)
 */
window.importFromXML = function(xmlString) {
  try {
    const parser = new DOMParser();
    const xmlDoc = parser.parseFromString(xmlString, 'text/xml');
    
    // Check for parsing errors
    const parserError = xmlDoc.querySelector('parsererror');
    if (parserError) {
      throw new Error('Invalid XML format: ' + parserError.textContent);
    }
    
    const root = xmlDoc.documentElement;
    
    if (root.tagName !== 'cite-linker-config') {
      throw new Error('Invalid XML format: root element should be cite-linker-config');
    }
    
    const result = {
      settings: {},
      patterns: [],
      metadata: {
        version: root.getAttribute('version') || '1.0',
        exported: root.getAttribute('exported') || ''
      }
    };
    
    // Parse settings
    const settingsEl = root.querySelector('settings');
    if (settingsEl) {
      const settingElements = settingsEl.querySelectorAll('setting');
      settingElements.forEach(settingEl => {
        const name = settingEl.getAttribute('name');
        const items = settingEl.querySelectorAll('item');
        
        if (items.length > 0) {
          // Array setting
          result.settings[name] = Array.from(items).map(item => item.textContent);
        } else {
          // Single value setting
          const value = settingEl.textContent;
          
          // Convert to appropriate type
          if (value === 'true' || value === 'false') {
            result.settings[name] = value === 'true';
          } else {
            result.settings[name] = value;
          }
        }
      });
    }
    
    // Parse patterns
    const patternsEl = root.querySelector('patterns');
    if (patternsEl) {
      const patternElements = patternsEl.querySelectorAll('pattern');
      patternElements.forEach(patternEl => {
        const pattern = {
          id: patternEl.getAttribute('id') || Date.now().toString(),
          enabled: patternEl.getAttribute('enabled') !== 'false',
          isPredefined: patternEl.getAttribute('predefined') === 'true',
          name: patternEl.querySelector('name')?.textContent || '',
          regex: patternEl.querySelector('regex')?.textContent || '',
          baseUrl: patternEl.querySelector('baseUrl')?.textContent || '',
          description: patternEl.querySelector('description')?.textContent || ''
        };
        
        // Validate pattern
        if (pattern.name && pattern.regex && pattern.baseUrl) {
          result.patterns.push(pattern);
        }
      });
    }
    
    return {
      success: true,
      data: result
    };
    
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
};

/**
 * Validate XML before import (supports both formats)
 */
window.validateXML = function(xmlString) {
  try {
    const parser = new DOMParser();
    const xmlDoc = parser.parseFromString(xmlString, 'text/xml');
    
    const parserError = xmlDoc.querySelector('parsererror');
    if (parserError) {
      return {
        valid: false,
        error: 'Invalid XML syntax: ' + parserError.textContent
      };
    }
    
    const root = xmlDoc.documentElement;
    
    // Check for RegexContainer format
    if (root.tagName === 'ArrayOfRegexContainer') {
      const regexContainers = root.querySelectorAll('RegexContainer');
      return {
        valid: true,
        format: 'RegexContainer',
        metadata: {
          format: 'RegexContainer',
          patternsCount: regexContainers.length,
          settingsCount: 0
        }
      };
    }
    
    // Check for cite-linker-config format
    if (root.tagName === 'cite-linker-config') {
      const patternsCount = root.querySelectorAll('patterns pattern').length;
      const settingsCount = root.querySelectorAll('settings setting').length;
      
      return {
        valid: true,
        format: 'cite-linker-config',
        metadata: {
          format: 'cite-linker-config',
          version: root.getAttribute('version') || '1.0',
          exported: root.getAttribute('exported') || '',
          patternsCount,
          settingsCount
        }
      };
    }
    
    return {
      valid: false,
      error: 'Invalid root element. Expected: cite-linker-config or ArrayOfRegexContainer'
    };
    
  } catch (error) {
    return {
      valid: false,
      error: error.message
    };
  }
};

/**
 * Format XML for readability
 */
function formatXML(xmlString) {
  const formatted = [];
  let indent = 0;
  
  xmlString.split(/>\s*</).forEach((node, index) => {
    if (index > 0) {
      node = '<' + node;
    }
    if (index < xmlString.split(/>\s*</).length - 1) {
      node = node + '>';
    }
    
    const padding = '  '.repeat(indent);
    
    if (node.match(/^<\/\w/)) {
      indent--;
      formatted.push('  '.repeat(indent) + node);
    } else if (node.match(/^<\w[^>]*[^\/]>$/)) {
      formatted.push('  '.repeat(indent) + node);
      indent++;
    } else {
      formatted.push('  '.repeat(indent) + node);
    }
  });
  
  return '<?xml version="1.0" encoding="UTF-8"?>\n' + formatted.join('\n');
}

/**
 * Download XML file
 */
window.downloadXML = function(xmlContent, filename = 'cite-linker-config.xml') {
  const blob = new Blob([xmlContent], { type: 'application/xml' });
  const url = URL.createObjectURL(blob);
  
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  
  URL.revokeObjectURL(url);
};

/**
 * Read XML file
 */
window.readXMLFile = function(file) {
  return new Promise((resolve, reject) => {
    if (!file) {
      reject(new Error('No file provided'));
      return;
    }
    
    if (!file.name.toLowerCase().endsWith('.xml')) {
      reject(new Error('File must have .xml extension'));
      return;
    }
    
    const reader = new FileReader();
    
    reader.onload = (e) => {
      resolve(e.target.result);
    };
    
    reader.onerror = () => {
      reject(new Error('Failed to read file'));
    };
    
    reader.readAsText(file);
  });
};

/**
 * Universal import function that detects format and imports accordingly
 */
window.importFromXMLAny = function(xmlString) {
  const validation = window.validateXML(xmlString);
  
  if (!validation.valid) {
    return {
      success: false,
      error: validation.error
    };
  }
  
  if (validation.format === 'RegexContainer') {
    return window.importFromRegexContainerXML(xmlString);
  } else if (validation.format === 'cite-linker-config') {
    return window.importFromXML(xmlString);
  } else {
    return {
      success: false,
      error: 'Unknown XML format'
    };
  }
};