/**
 * Validation utilities for patterns and URLs
 */

/**
 * Validate regex pattern
 */
window.validateRegex = function(pattern) {
  const errors = [];
  
  if (!pattern || typeof pattern !== 'string') {
    errors.push('Pattern cannot be empty');
    return { isValid: false, errors };
  }
  
  try {
    new RegExp(pattern);
  } catch (error) {
    errors.push(`Invalid regex: ${error.message}`);
  }
  
  // Check for capturing groups
  const groupCount = (pattern.match(/(?<!\\)\(/g) || []).length;
  if (groupCount === 0) {
    errors.push('Pattern should contain at least one capturing group for URL substitution');
  }
  
  return {
    isValid: errors.length === 0,
    errors,
    groupCount
  };
}

/**
 * Validate base URL with placeholders
 */
window.validateBaseUrl = function(url, expectedGroups = 0) {
  const errors = [];
  
  if (!url || typeof url !== 'string') {
    errors.push('Base URL cannot be empty');
    return { isValid: false, errors };
  }
  
  // Basic URL validation
  try {
    const urlObj = new URL(url.replace(/\{\d+\}/g, 'test').replace(/\$\d+/g, 'test'));
    if (!['http:', 'https:'].includes(urlObj.protocol)) {
      errors.push('URL must use http or https protocol');
    }
  } catch (error) {
    errors.push('Invalid URL format');
  }
  
  // Check for placeholder groups (both {n} and $n formats)
  const bracePlaceholders = url.match(/\{(\d+)\}/g) || [];
  const dollarPlaceholders = url.match(/\$(\d+)/g) || [];
  const allPlaceholders = [...bracePlaceholders, ...dollarPlaceholders];
  const groupNumbers = allPlaceholders.map(p => {
    if (p.startsWith('{')) {
      return parseInt(p.substring(1, p.length - 1));
    } else {
      return parseInt(p.substring(1));
    }
  });
  
  if (expectedGroups > 0) {
    const maxGroup = Math.max(...groupNumbers, 0);
    if (maxGroup > expectedGroups) {
      errors.push(`URL references group $${maxGroup} but pattern only has ${expectedGroups} groups`);
    }
    
    // Check for missing sequential groups
    for (let i = 1; i <= expectedGroups; i++) {
      if (!groupNumbers.includes(i) && allPlaceholders.length > 0) {
        // This is a warning, not an error
      }
    }
  }
  
  return {
    isValid: errors.length === 0,
    errors,
    placeholders: groupNumbers
  };
}

/**
 * Validate complete pattern configuration
 */
window.validatePattern = function(pattern) {
  const errors = [];
  
  if (!pattern.name || pattern.name.trim().length === 0) {
    errors.push('Pattern name is required');
  }
  
  const regexValidation = window.validateRegex(pattern.regex);
  if (!regexValidation.isValid) {
    errors.push(...regexValidation.errors);
  }
  
  const urlValidation = window.validateBaseUrl(pattern.baseUrl, regexValidation.groupCount);
  if (!urlValidation.isValid) {
    errors.push(...urlValidation.errors);
  }
  
  return {
    isValid: errors.length === 0,
    errors,
    regexValidation,
    urlValidation
  };
}

/**
 * Validate domain list
 */
window.validateDomains = function(domains) {
  const errors = [];
  const validDomains = [];
  
  if (!Array.isArray(domains)) {
    errors.push('Domains must be an array');
    return { isValid: false, errors, validDomains };
  }
  
  domains.forEach((domain, index) => {
    if (typeof domain !== 'string' || domain.trim().length === 0) {
      errors.push(`Domain at index ${index} is invalid`);
      return;
    }
    
    const cleanDomain = domain.trim().toLowerCase();
    
    // Basic domain validation
    if (!/^[a-z0-9.-]+$/.test(cleanDomain)) {
      errors.push(`Invalid domain format: ${domain}`);
      return;
    }
    
    if (cleanDomain.startsWith('.') || cleanDomain.endsWith('.')) {
      errors.push(`Domain cannot start or end with dot: ${domain}`);
      return;
    }
    
    validDomains.push(cleanDomain);
  });
  
  return {
    isValid: errors.length === 0,
    errors,
    validDomains
  };
}

/**
 * Test regex pattern against sample text
 */
window.testPattern = function(pattern, sampleText) {
  try {
    const regex = new RegExp(pattern, 'g');
    const matches = [];
    let match;
    
    while ((match = regex.exec(sampleText)) !== null) {
      matches.push({
        fullMatch: match[0],
        groups: match.slice(1),
        index: match.index
      });
      
      // Prevent infinite loop on zero-length matches
      if (match.index === regex.lastIndex) {
        regex.lastIndex++;
      }
    }
    
    return {
      success: true,
      matches
    };
  } catch (error) {
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Generate URL from pattern and match groups
 */
window.generateUrl = function(baseUrl, groups) {
  try {
    let url = baseUrl;
    groups.forEach((group, index) => {
      // Support both {n} and $n placeholder formats
      // For {n} format (0-based indexing)
      const bracePattern = new RegExp('\\{' + index + '\\}', 'g');
      url = url.replace(bracePattern, encodeURIComponent(group));
      
      // For $n format (1-based indexing)  
      const dollarPattern = new RegExp('\\$' + (index + 1), 'g');
      url = url.replace(dollarPattern, encodeURIComponent(group));
    });
    return url;
  } catch (error) {
    console.error('Error generating URL:', error, 'baseUrl:', baseUrl, 'groups:', groups);
    return null;
  }
}