/**
 * Storage utilities for Cite Linker Pro extension
 */

window.STORAGE_KEYS = {
  ENABLED: 'enabled',
  PATTERNS: 'patterns',
  PATTERNS_CHUNKS: 'patterns_chunks', // For chunked storage
  PATTERNS_METADATA: 'patterns_metadata', // Chunk metadata
  DOMAIN_MODE: 'domainMode', // 'all', 'whitelist', 'blacklist'
  DOMAINS: 'domains',
  LINK_STYLE: 'linkStyle',
  SHOW_TOOLTIPS: 'showTooltips'
};

// Storage configuration
window.STORAGE_CONFIG = {
  MAX_CHUNK_SIZE: 7000, // Leave buffer below 8192 byte limit
  PATTERNS_PER_CHUNK: 15 // Approximate patterns per chunk
};

window.DEFAULT_SETTINGS = {
  [STORAGE_KEYS.ENABLED]: true,
  [window.STORAGE_KEYS.PATTERNS]: [],
  [STORAGE_KEYS.DOMAIN_MODE]: 'all',
  [STORAGE_KEYS.DOMAINS]: [],
  [STORAGE_KEYS.LINK_STYLE]: 'dashed',
  [STORAGE_KEYS.SHOW_TOOLTIPS]: true,
  autoScan: false, // Default to manual scanning for new installs
  maxPatterns: 250
};

/**
 * Get settings from chrome.storage.sync
 */
window.getSettings = async function() {
  try {
    const result = await chrome.storage.sync.get(window.DEFAULT_SETTINGS);
    return result;
  } catch (error) {
    console.error('Error getting settings:', error);
    return window.DEFAULT_SETTINGS;
  }
}

/**
 * Save settings to chrome.storage.sync
 */
window.saveSettings = async function(settings) {
  try {
    await chrome.storage.sync.set(settings);
    return true;
  } catch (error) {
    console.error('Error saving settings:', error);
    return false;
  }
}

/**
 * Get specific setting value
 */
window.getSetting = async function(key) {
  try {
    // Handle backward compatibility for patterns
    if (key === window.STORAGE_KEYS.PATTERNS) {
      return await window.loadPatternsChunked();
    }
    
    const result = await chrome.storage.sync.get({ [key]: window.DEFAULT_SETTINGS[key] });
    return result[key];
  } catch (error) {
    console.error(`Error getting setting ${key}:`, error);
    return window.DEFAULT_SETTINGS[key];
  }
}

/**
 * Migrate old pattern storage to chunked storage
 */
window.migratePatternStorage = async function() {
  try {
    // Check if old patterns exist
    const result = await chrome.storage.sync.get(['patterns']);
    const oldPatterns = result.patterns;
    
    if (oldPatterns && Array.isArray(oldPatterns) && oldPatterns.length > 0) {
      console.log('Migrating', oldPatterns.length, 'patterns to chunked storage...');
      
      // Save using new chunked system
      const success = await window.savePatternsChunked(oldPatterns);
      
      if (success) {
        // Remove old patterns key
        await chrome.storage.sync.remove(['patterns']);
        console.log('Migration completed successfully');
        return { success: true, migrated: oldPatterns.length };
      } else {
        console.error('Failed to save patterns during migration');
        return { success: false, error: 'Failed to save patterns' };
      }
    }
    
    return { success: true, migrated: 0 };
  } catch (error) {
    console.error('Error during pattern migration:', error);
    return { success: false, error: error.message };
  }
}

/**
 * Save specific setting value
 */
window.saveSetting = async function(key, value) {
  try {
    await chrome.storage.sync.set({ [key]: value });
    return true;
  } catch (error) {
    console.error(`Error saving setting ${key}:`, error);
    return false;
  }
}

/**
 * Split patterns into chunks to avoid storage quotas
 */
window.chunkPatterns = function(patterns) {
  const chunks = [];
  for (let i = 0; i < patterns.length; i += window.STORAGE_CONFIG.PATTERNS_PER_CHUNK) {
    chunks.push(patterns.slice(i, i + window.STORAGE_CONFIG.PATTERNS_PER_CHUNK));
  }
  return chunks;
}

/**
 * Save patterns using chunked storage
 */
window.savePatternsChunked = async function(patterns) {
  try {
    // Clear existing pattern chunks
    await window.clearPatternChunks();
    
    if (patterns.length === 0) {
      await chrome.storage.sync.set({
        [window.STORAGE_KEYS.PATTERNS_METADATA]: { chunkCount: 0, totalPatterns: 0 }
      });
      return true;
    }

    const chunks = window.chunkPatterns(patterns);
    const chunkData = {};
    
    // Save each chunk
    for (let i = 0; i < chunks.length; i++) {
      const chunkKey = `${window.STORAGE_KEYS.PATTERNS_CHUNKS}_${i}`;
      chunkData[chunkKey] = chunks[i];
    }
    
    // Save metadata
    chunkData[window.STORAGE_KEYS.PATTERNS_METADATA] = {
      chunkCount: chunks.length,
      totalPatterns: patterns.length,
      lastUpdated: new Date().toISOString()
    };
    
    await chrome.storage.sync.set(chunkData);
    return true;
  } catch (error) {
    console.error('Error saving patterns with chunks:', error);
    return false;
  }
}

/**
 * Load patterns from chunked storage
 */
window.loadPatternsChunked = async function() {
  try {
    // Get metadata first
    const result = await chrome.storage.sync.get([window.STORAGE_KEYS.PATTERNS_METADATA]);
    const metadata = result[window.STORAGE_KEYS.PATTERNS_METADATA];
    
    if (!metadata || metadata.chunkCount === 0) {
      return [];
    }
    
    // Load all chunks
    const chunkKeys = [];
    for (let i = 0; i < metadata.chunkCount; i++) {
      chunkKeys.push(`${window.STORAGE_KEYS.PATTERNS_CHUNKS}_${i}`);
    }
    
    const chunksResult = await chrome.storage.sync.get(chunkKeys);
    const patterns = [];
    
    for (let i = 0; i < metadata.chunkCount; i++) {
      const chunkKey = `${window.STORAGE_KEYS.PATTERNS_CHUNKS}_${i}`;
      const chunk = chunksResult[chunkKey];
      if (chunk && Array.isArray(chunk)) {
        patterns.push(...chunk);
      }
    }
    
    return patterns;
  } catch (error) {
    console.error('Error loading patterns from chunks:', error);
    return [];
  }
}

/**
 * Clear all pattern chunks
 */
window.clearPatternChunks = async function() {
  try {
    const result = await chrome.storage.sync.get([window.STORAGE_KEYS.PATTERNS_METADATA]);
    const metadata = result[window.STORAGE_KEYS.PATTERNS_METADATA];
    
    if (metadata && metadata.chunkCount > 0) {
      const keysToRemove = [];
      for (let i = 0; i < metadata.chunkCount; i++) {
        keysToRemove.push(`${window.STORAGE_KEYS.PATTERNS_CHUNKS}_${i}`);
      }
      await chrome.storage.sync.remove(keysToRemove);
    }
  } catch (error) {
    console.error('Error clearing pattern chunks:', error);
  }
}

/**
 * Add a new pattern to storage
 */
window.addPattern = async function(pattern) {
  try {
    const patterns = await window.loadPatternsChunked();
    const updatedPatterns = [...patterns, { ...pattern, id: Date.now().toString() }];
    return await window.savePatternsChunked(updatedPatterns);
  } catch (error) {
    console.error('Error adding pattern:', error);
    return false;
  }
}

/**
 * Update existing pattern
 */
window.updatePattern = async function(patternId, updatedPattern) {
  try {
    const patterns = await window.loadPatternsChunked();
    const updatedPatterns = patterns.map(p => 
      p.id === patternId ? { ...p, ...updatedPattern } : p
    );
    return await window.savePatternsChunked(updatedPatterns);
  } catch (error) {
    console.error('Error updating pattern:', error);
    return false;
  }
}

/**
 * Delete pattern
 */
window.deletePattern = async function(patternId) {
  try {
    const patterns = await window.loadPatternsChunked();
    const updatedPatterns = patterns.filter(p => p.id !== patternId);
    return await window.savePatternsChunked(updatedPatterns);
  } catch (error) {
    console.error('Error deleting pattern:', error);
    return false;
  }
}

/**
 * Reorder patterns by providing new array of pattern IDs
 */
window.reorderPatterns = async function(orderedPatternIds) {
  try {
    const patterns = await window.loadPatternsChunked();
    const reorderedPatterns = [];
    
    // Build new array in the specified order
    orderedPatternIds.forEach(id => {
      const pattern = patterns.find(p => p.id === id);
      if (pattern) {
        reorderedPatterns.push(pattern);
      }
    });
    
    // Add any patterns that weren't in the ordered list (shouldn't happen normally)
    patterns.forEach(pattern => {
      if (!orderedPatternIds.includes(pattern.id)) {
        reorderedPatterns.push(pattern);
      }
    });
    
    return await window.savePatternsChunked(reorderedPatterns);
  } catch (error) {
    console.error('Error reordering patterns:', error);
    return false;
  }
}