/**
 * Popup script for Cite Linker Pro
 */

// Storage utilities will be loaded from content script

class PopupManager {
  constructor() {
    this.currentTab = null;
    this.settings = {};
    this.init();
  }
  
  async init() {
    try {
      await this.loadCurrentTab();
      await this.loadSettings();
      this.setupEventListeners();
      this.updateUI();
      this.loadStats();
    } catch (error) {
      console.error('Popup initialization failed:', error);
      this.showError('Failed to initialize popup');
    }
  }
  
  async loadCurrentTab() {
    try {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      this.currentTab = tabs[0];
    } catch (error) {
      console.error('Failed to get current tab:', error);
    }
  }
  
  async loadSettings() {
    try {
      this.settings = await chrome.storage.sync.get({
        enabled: true,
        patterns: [],
        domainMode: 'all',
        domains: [],
        linkStyle: 'dashed',
        showTooltips: true,
        autoScan: false
      });
    } catch (error) {
      console.error('Failed to load settings:', error);
      this.settings = {};
    }
  }
  
  setupEventListeners() {
    // Toggle button
    const toggleBtn = document.getElementById('toggleBtn');
    toggleBtn?.addEventListener('click', this.handleToggle.bind(this));
    
    // Process page button
    const processPageBtn = document.getElementById('processPageBtn');
    processPageBtn?.addEventListener('click', this.handleProcessPage.bind(this));
    
    // Process selection button
    const processSelectionBtn = document.getElementById('processSelectionBtn');
    processSelectionBtn?.addEventListener('click', this.handleProcessSelection.bind(this));
    
    // Settings button
    const settingsBtn = document.getElementById('settingsBtn');
    settingsBtn?.addEventListener('click', this.handleSettings.bind(this));
    
    // Auto-scan checkbox
    const autoScanCheckbox = document.getElementById('autoScanCheckbox');
    autoScanCheckbox?.addEventListener('change', this.handleAutoScanToggle.bind(this));
    
    // Listen for storage changes
    chrome.storage.onChanged.addListener((changes) => {
      this.handleStorageChange(changes);
    });
  }
  
  updateUI() {
    this.updateStatus();
    this.updateDomainInfo();
    this.updateToggleButton();
    this.updateAutoScanCheckbox();
  }
  
  updateStatus() {
    const statusDot = document.getElementById('statusDot');
    const statusText = document.getElementById('statusText');
    
    if (!statusDot || !statusText) return;
    
    const isEnabled = this.settings.enabled;
    const isAllowed = this.isDomainAllowed();
    
    statusDot.className = 'status-dot';
    
    if (isEnabled && isAllowed) {
      statusDot.classList.add('enabled');
      statusText.textContent = 'Active';
    } else if (isEnabled && !isAllowed) {
      statusDot.classList.add('disabled');
      statusText.textContent = 'Domain Blocked';
    } else {
      statusDot.classList.add('disabled');
      statusText.textContent = 'Disabled';
    }
  }
  
  updateDomainInfo() {
    const currentDomainEl = document.getElementById('currentDomain');
    const domainStatus = document.getElementById('domainStatus');
    
    if (!currentDomainEl || !domainStatus) return;
    
    const domain = this.currentTab?.url ? new URL(this.currentTab.url).hostname : 'Unknown';
    currentDomainEl.textContent = domain;
    
    const isAllowed = this.isDomainAllowed();
    const statusText = domainStatus.querySelector('.domain-status-text');
    
    domainStatus.className = 'domain-status';
    if (statusText) {
      statusText.textContent = isAllowed ? 'Allowed' : 'Blocked';
    }
    
    if (!isAllowed) {
      domainStatus.classList.add('blocked');
    }
  }
  
  updateToggleButton() {
    const toggleBtn = document.getElementById('toggleBtn');
    const toggleIcon = document.getElementById('toggleIcon');
    const toggleText = document.getElementById('toggleText');
    
    if (!toggleBtn || !toggleIcon || !toggleText) return;
    
    const isEnabled = this.settings.enabled;
    
    toggleBtn.className = 'toggle-btn';
    if (isEnabled) {
      toggleIcon.textContent = '⏸️';
      toggleText.textContent = 'Disable';
    } else {
      toggleBtn.classList.add('disabled');
      toggleIcon.textContent = '▶️';
      toggleText.textContent = 'Enable';
    }
  }
  
  async loadStats() {
    try {
      if (!this.currentTab) return;
      
      // Send message to content script to get stats
      const response = await chrome.tabs.sendMessage(this.currentTab.id, {
        type: 'GET_STATS'
      });
      
      if (response?.success) {
        this.updateStats(response.stats);
      } else {
        this.updateStats({ citations: 0, patterns: 0 });
      }
    } catch (error) {
      // Content script might not be loaded
      this.updateStats({ citations: 0, patterns: 0 });
    }
    
    // Update pattern count from settings
    const patterns = this.settings.patterns || [];
    const enabledPatterns = patterns.filter(p => p.enabled !== false).length;
    
    const patternCountEl = document.getElementById('patternCount');
    if (patternCountEl) {
      patternCountEl.textContent = enabledPatterns.toString();
    }
  }
  
  updateStats(stats) {
    const citationCountEl = document.getElementById('citationCount');
    
    if (citationCountEl) {
      citationCountEl.textContent = (stats.citations || 0).toString();
    }
  }
  
  isDomainAllowed() {
    if (!this.currentTab?.url) return false;
    
    try {
      const currentDomain = new URL(this.currentTab.url).hostname.toLowerCase();
      const domainMode = this.settings.domainMode || 'all';
      const domains = this.settings.domains || [];
      
      switch (domainMode) {
        case 'whitelist':
          return domains.some(domain => 
            currentDomain === domain || currentDomain.endsWith('.' + domain)
          );
          
        case 'blacklist':
          return !domains.some(domain => 
            currentDomain === domain || currentDomain.endsWith('.' + domain)
          );
          
        case 'all':
        default:
          return true;
      }
    } catch (error) {
      return false;
    }
  }
  
  async handleToggle() {
    try {
      const newState = !this.settings.enabled;
      await chrome.storage.sync.set({ enabled: newState });
      
      // Send message to content script
      if (this.currentTab) {
        chrome.tabs.sendMessage(this.currentTab.id, {
          type: 'TOGGLE_EXTENSION'
        }).catch(() => {
          // Content script might not be loaded, ignore error
        });
      }
      
      // Update local settings
      this.settings.enabled = newState;
      this.updateUI();
      
    } catch (error) {
      console.error('Failed to toggle extension:', error);
      this.showError('Failed to toggle extension');
    }
  }
  
  async handleProcessPage() {
    try {
      if (!this.currentTab) return;
      
      const processPageBtn = document.getElementById('processPageBtn');
      if (processPageBtn) {
        processPageBtn.disabled = true;
        processPageBtn.innerHTML = '<span class="btn-icon">⏳</span>Processing...';
      }
      
      await chrome.tabs.sendMessage(this.currentTab.id, {
        type: 'PROCESS_PAGE'
      });
      
      // Reload stats after processing
      setTimeout(() => {
        this.loadStats();
        if (processPageBtn) {
          processPageBtn.disabled = false;
          processPageBtn.innerHTML = '<span class="btn-icon">🔄</span>Scan Page';
        }
      }, 1000);
      
    } catch (error) {
      console.error('Failed to process page:', error);
      this.showError('Failed to process page');
      
      const processPageBtn = document.getElementById('processPageBtn');
      if (processPageBtn) {
        processPageBtn.disabled = false;
        processPageBtn.innerHTML = '<span class="btn-icon">🔄</span>Scan Page';
      }
    }
  }
  
  async handleProcessSelection() {
    try {
      if (!this.currentTab) return;
      
      await chrome.tabs.sendMessage(this.currentTab.id, {
        type: 'PROCESS_SELECTION'
      });
      
      // Close popup after processing selection
      window.close();
      
    } catch (error) {
      console.error('Failed to process selection:', error);
      this.showError('Failed to process selection');
    }
  }
  
  handleSettings() {
    chrome.runtime.openOptionsPage();
    window.close();
  }
  
  handleStorageChange(changes) {
    if (changes.enabled || 
        changes.domainMode || 
        changes.domains || 
        changes.autoScan) {
      this.loadSettings().then(() => {
        this.updateUI();
      });
    }
    
    if (changes.patterns) {
      this.loadStats();
    }
  }
  
  updateAutoScanCheckbox() {
    const autoScanCheckbox = document.getElementById('autoScanCheckbox');
    if (autoScanCheckbox) {
      autoScanCheckbox.checked = this.settings.autoScan || false;
    }
  }
  
  async handleAutoScanToggle() {
    try {
      const autoScanCheckbox = document.getElementById('autoScanCheckbox');
      const newAutoScan = autoScanCheckbox?.checked || false;
      
      await chrome.storage.sync.set({ autoScan: newAutoScan });
      
      // Send message to content script
      if (this.currentTab) {
        chrome.tabs.sendMessage(this.currentTab.id, {
          type: 'UPDATE_AUTO_SCAN',
          autoScan: newAutoScan
        }).catch(() => {
          // Content script might not be loaded, ignore error
        });
      }
      
      // Update local settings
      this.settings.autoScan = newAutoScan;
      
    } catch (error) {
      console.error('Failed to toggle auto-scan:', error);
      this.showError('Failed to update auto-scan setting');
    }
  }
  
  showError(message) {
    // Simple error display - could be enhanced with a toast or modal
    const statusText = document.getElementById('statusText');
    if (statusText) {
      const originalText = statusText.textContent;
      statusText.textContent = 'Error';
      statusText.style.color = '#f44336';
      
      setTimeout(() => {
        statusText.textContent = originalText;
        statusText.style.color = '';
      }, 3000);
    }
  }
}

// Initialize popup when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  new PopupManager();
});