/**
 * Enhanced Options Page Script for Cite Linker Pro
 */

class OptionsManager {
  constructor() {
    this.settings = {};
    this.patterns = [];
    this.predefinedPatterns = [];
    this.currentEditingPattern = null;
    this.init();
  }
  
  async init() {
    try {
      // Migrate old pattern storage if needed
      const migration = await window.migratePatternStorage();
      if (migration.migrated > 0) {
        this.showStatus(`Migrated ${migration.migrated} patterns to new storage system`, 'success');
      }
      
      await this.loadSettings();
      await this.loadPatterns();
      this.setupEventListeners();
      this.setupTabs();
      this.renderPatterns();
      this.updateUI();
    } catch (error) {
      console.error('Options initialization failed:', error);
      this.showStatus('Failed to initialize settings', 'error');
    }
  }
  
  async loadSettings() {
    try {
      this.settings = await chrome.storage.sync.get({
        enabled: true,
        linkStyle: 'dashed',
        showTooltips: true,
        domainMode: 'all',
        domains: [],
        patterns: [],
        maxPatterns: 250,
        autoScan: false
      });
    } catch (error) {
      console.error('Failed to load settings:', error);
      this.settings = {};
    }
  }
  
  async loadPatterns() {
    try {
      // Initialize and load predefined patterns
      if (window.initializePredefinedPatterns) {
        await window.initializePredefinedPatterns();
      }
      
      // Load predefined patterns (now async)
      this.predefinedPatterns = window.getPredefinedPatterns ? 
        await window.getPredefinedPatterns() : 
        [];
      
      // Load user patterns from chunked storage
      const userPatterns = await window.loadPatternsChunked();
      
      // Combine all patterns
      this.patterns = [...this.predefinedPatterns, ...userPatterns];
      
      // Populate test pattern dropdown
      this.updateTestPatternDropdown();
    } catch (error) {
      console.error('Failed to load patterns:', error);
      // Use fallback patterns if available
      this.predefinedPatterns = window.getPredefinedPatternsSync ? 
        window.getPredefinedPatternsSync() : 
        [];
      this.patterns = [...this.predefinedPatterns];
    }
  }
  
  setupEventListeners() {
    // General settings
    document.getElementById('enabled').addEventListener('change', () => this.updateSetting('enabled'));
    document.getElementById('linkStyle').addEventListener('change', () => this.updateSetting('linkStyle'));
    document.getElementById('showTooltips').addEventListener('change', () => this.updateSetting('showTooltips'));
    document.getElementById('maxPatterns').addEventListener('input', () => this.updateSetting('maxPatterns'));
    document.getElementById('autoScan').addEventListener('change', () => this.updateSetting('autoScan'));
    document.getElementById('domainMode').addEventListener('change', () => this.updateSetting('domainMode'));
    document.getElementById('domains').addEventListener('input', () => this.updateSetting('domains'));
    
    // Event delegation for pattern action buttons
    document.getElementById('patternsList').addEventListener('click', (e) => {
      // First check if this is a button click (highest priority)
      const patternId = e.target.getAttribute('data-pattern-id');
      if (patternId) {
        if (e.target.classList.contains('pattern-toggle')) {
          this.togglePattern(patternId);
          return;
        } else if (e.target.textContent === 'Edit') {
          this.editPattern(patternId);
          return;
        } else if (e.target.textContent === 'Delete') {
          this.deletePattern(patternId);
          return;
        }
      }
      
      // Handle pattern header clicks for expanding/collapsing
      // Only if we're not clicking on an action button
      if (e.target.closest('[data-toggle-target]') && !e.target.closest('[data-stop-propagation]')) {
        const header = e.target.closest('[data-toggle-target]');
        const details = header.nextElementSibling;
        if (details) {
          details.classList.toggle('expanded');
        }
      }
    });
    
    // Action buttons
    document.getElementById('saveBtn').addEventListener('click', () => this.saveAllSettings());
    document.getElementById('resetBtn').addEventListener('click', () => this.resetSettings());
    document.getElementById('reloadPatternsBtn').addEventListener('click', () => this.reloadPredefinedPatterns());
    
    // Pattern management
    document.getElementById('addPatternBtn').addEventListener('click', () => this.showPatternModal());
    document.getElementById('patternSearch').addEventListener('input', () => this.filterPatterns());
    document.getElementById('showPredefined').addEventListener('change', () => this.filterPatterns());
    document.getElementById('showCustom').addEventListener('change', () => this.filterPatterns());
    document.getElementById('showEnabled').addEventListener('change', () => this.filterPatterns());
    
    // Import/Export
    document.getElementById('exportBtn').addEventListener('click', () => this.exportConfig());
    document.getElementById('importBtn').addEventListener('click', () => this.triggerImport());
    document.getElementById('importFile').addEventListener('change', (e) => this.importConfig(e));
    
    // Preview and testing
    document.getElementById('testPatternBtn').addEventListener('click', () => this.testSelectedPattern());
    document.getElementById('previewAllBtn').addEventListener('click', () => this.previewAllPatterns());
    document.getElementById('testText').addEventListener('input', () => this.updateLivePreview());
    
    // Modal events
    document.getElementById('savePatternBtn').addEventListener('click', () => this.savePattern());
    document.getElementById('cancelPatternBtn').addEventListener('click', () => this.hidePatternModal());
    document.querySelector('.close').addEventListener('click', () => this.hidePatternModal());
    
    // Pattern validation
    document.getElementById('patternRegex').addEventListener('input', () => this.validateCurrentPattern());
    document.getElementById('patternUrl').addEventListener('input', () => this.validateCurrentPattern());
    document.getElementById('sampleText').addEventListener('input', () => this.updatePatternTesting());
    
    // Close modal on outside click
    document.getElementById('patternModal').addEventListener('click', (e) => {
      if (e.target.id === 'patternModal') {
        this.hidePatternModal();
      }
    });
  }
  
  setupTabs() {
    const tabBtns = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const targetTab = btn.dataset.tab;
        
        // Update tab buttons
        tabBtns.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        
        // Update tab contents
        tabContents.forEach(content => {
          content.classList.remove('active');
          if (content.id === targetTab + '-tab') {
            content.classList.add('active');
          }
        });
        
        // Special handling for preview tab
        if (targetTab === 'preview') {
          this.updateTestPatternDropdown();
          this.updateLivePreview();
        }
      });
    });
  }
  
  updateUI() {
    // Update form controls with current settings
    document.getElementById('enabled').checked = this.settings.enabled;
    document.getElementById('linkStyle').value = this.settings.linkStyle || 'dashed';
    document.getElementById('showTooltips').checked = this.settings.showTooltips !== false;
    document.getElementById('maxPatterns').value = this.settings.maxPatterns || 250;
    document.getElementById('autoScan').checked = this.settings.autoScan === true;
    document.getElementById('domainMode').value = this.settings.domainMode || 'all';
    document.getElementById('domains').value = (this.settings.domains || []).join('\n');
  }
  
  async updateSetting(key) {
    const element = document.getElementById(key);
    let value;
    
    if (element.type === 'checkbox') {
      value = element.checked;
    } else if (key === 'domains') {
      value = element.value.split('\n').map(d => d.trim()).filter(d => d);
    } else if (key === 'maxPatterns') {
      value = parseInt(element.value) || 250;
    } else {
      value = element.value;
    }
    
    this.settings[key] = value;
    
    try {
      await chrome.storage.sync.set({ [key]: value });
    } catch (error) {
      console.error(`Failed to save setting ${key}:`, error);
    }
  }
  
  async saveAllSettings() {
    try {
      await chrome.storage.sync.set(this.settings);
      this.showStatus('All settings saved successfully!', 'success');
    } catch (error) {
      console.error('Failed to save settings:', error);
      this.showStatus('Failed to save settings', 'error');
    }
  }
  
  async resetSettings() {
    if (!confirm('Are you sure you want to reset all settings to defaults? This will remove all custom patterns.')) {
      return;
    }
    
    try {
      await chrome.storage.sync.clear();
      await this.loadSettings();
      await this.loadPatterns();
      this.updateUI();
      this.renderPatterns();
      this.showStatus('Settings reset to defaults', 'success');
    } catch (error) {
      console.error('Failed to reset settings:', error);
      this.showStatus('Failed to reset settings', 'error');
    }
  }
  
  async reloadPredefinedPatterns() {
    try {
      // Clear cache to force reload from XML
      if (window.clearPredefinedPatternsCache) {
        window.clearPredefinedPatternsCache();
      }
      
      await this.loadPatterns();
      this.renderPatterns();
      this.showStatus('Predefined patterns reloaded', 'success');
    } catch (error) {
      console.error('Failed to reload patterns:', error);
      this.showStatus('Failed to reload patterns', 'error');
    }
  }
  
  renderPatterns() {
    const container = document.getElementById('patternsList');
    container.innerHTML = '';
    
    const filteredPatterns = this.getFilteredPatterns();
    
    if (filteredPatterns.length === 0) {
      container.innerHTML = '<div class="no-matches">No patterns match your filters</div>';
      return;
    }
    
    filteredPatterns.forEach((pattern, index) => {
      const patternEl = this.createPatternElement(pattern, index);
      container.appendChild(patternEl);
    });
  }
  
  createPatternElement(pattern, index) {
    const div = document.createElement('div');
    div.className = 'pattern-item';
    div.draggable = true;
    div.dataset.patternId = pattern.id;
    div.dataset.patternIndex = index;
    
    const badges = [];
    if (pattern.isPredefined) {
      badges.push('<span class="pattern-badge predefined">Predefined</span>');
    } else {
      badges.push('<span class="pattern-badge custom">Custom</span>');
    }
    
    if (pattern.enabled === false) {
      badges.push('<span class="pattern-badge disabled">Disabled</span>');
    }
    
    div.innerHTML = `
      <div class="pattern-header" data-toggle-target="true">
        <div class="pattern-order-number">${index + 1}</div>
        <div class="pattern-drag-handle">⋮⋮</div>
        <div class="pattern-info">
          <div class="pattern-name">${this.escapeHtml(pattern.name)}</div>
          <div class="pattern-meta">
            ${badges.join(' ')}
            ${pattern.description ? `• ${this.escapeHtml(pattern.description)}` : ''}
          </div>
        </div>
        <div class="pattern-actions" data-stop-propagation="true">
          <button class="pattern-toggle ${pattern.enabled !== false ? 'enabled' : 'disabled'}" 
                  data-pattern-id="${pattern.id}">
            ${pattern.enabled !== false ? 'Enabled' : 'Disabled'}
          </button>
          ${!pattern.isPredefined ? `
            <button class="btn btn-primary" data-pattern-id="${pattern.id}">Edit</button>
            <button class="btn btn-danger" data-pattern-id="${pattern.id}">Delete</button>
          ` : ''}
        </div>
      </div>
      <div class="pattern-details">
        <div class="pattern-field">
          <label>Regular Expression:</label>
          <code>${this.escapeHtml(pattern.regex)}</code>
        </div>
        <div class="pattern-field">
          <label>Base URL:</label>
          <code>${this.escapeHtml(pattern.baseUrl)}</code>
        </div>
      </div>
    `;
    
    // Add drag event listeners
    div.addEventListener('dragstart', (e) => this.handleDragStart(e));
    div.addEventListener('dragover', (e) => this.handleDragOver(e));
    div.addEventListener('drop', (e) => this.handleDrop(e));
    div.addEventListener('dragend', (e) => this.handleDragEnd(e));
    
    return div;
  }
  
  getFilteredPatterns() {
    const searchTerm = document.getElementById('patternSearch').value.toLowerCase();
    const showPredefined = document.getElementById('showPredefined').checked;
    const showCustom = document.getElementById('showCustom').checked;
    const showEnabledOnly = document.getElementById('showEnabled').checked;
    
    return this.patterns.filter(pattern => {
      // Search filter
      if (searchTerm && !pattern.name.toLowerCase().includes(searchTerm) && 
          !pattern.description?.toLowerCase().includes(searchTerm)) {
        return false;
      }
      
      // Type filter
      if (pattern.isPredefined && !showPredefined) return false;
      if (!pattern.isPredefined && !showCustom) return false;
      
      // Enabled filter
      if (showEnabledOnly && pattern.enabled === false) return false;
      
      return true;
    });
  }
  
  filterPatterns() {
    this.renderPatterns();
  }
  
  async togglePattern(patternId) {
    const pattern = this.patterns.find(p => p.id === patternId);
    if (!pattern) return;
    
    pattern.enabled = !pattern.enabled;
    
    if (!pattern.isPredefined) {
      // Save custom pattern
      await this.saveUserPatterns();
    }
    
    this.renderPatterns();
  }
  
  showPatternModal(pattern = null) {
    this.currentEditingPattern = pattern;
    const modal = document.getElementById('patternModal');
    const title = document.getElementById('modalTitle');
    
    if (pattern) {
      title.textContent = 'Edit Pattern';
      document.getElementById('patternName').value = pattern.name || '';
      document.getElementById('patternRegex').value = pattern.regex || '';
      document.getElementById('patternUrl').value = pattern.baseUrl || '';
      document.getElementById('patternDescription').value = pattern.description || '';
      document.getElementById('patternEnabled').checked = pattern.enabled !== false;
    } else {
      title.textContent = 'Add New Pattern';
      document.getElementById('patternName').value = '';
      document.getElementById('patternRegex').value = '';
      document.getElementById('patternUrl').value = '';
      document.getElementById('patternDescription').value = '';
      document.getElementById('patternEnabled').checked = true;
    }
    
    // Initialize sample text if not already set
    const sampleTextEl = document.getElementById('sampleText');
    if (!sampleTextEl.value.trim()) {
      sampleTextEl.value = `410 U.S. 113 (Roe v. Wade)
42 U.S.C. § 1983 (Civil Rights Act)
123 F.2d 456 (Federal Reporter)
2023 ND 145 (North Dakota Supreme Court)
319 F.3d 1018 (Federal Court of Appeals)`;
    }
    
    modal.style.display = 'block';
    this.validateCurrentPattern();
  }
  
  hidePatternModal() {
    document.getElementById('patternModal').style.display = 'none';
    this.currentEditingPattern = null;
  }
  
  editPattern(patternId) {
    const pattern = this.patterns.find(p => p.id === patternId);
    if (pattern && !pattern.isPredefined) {
      this.showPatternModal(pattern);
    }
  }
  
  async deletePattern(patternId) {
    if (!confirm('Are you sure you want to delete this pattern?')) {
      return;
    }
    
    this.patterns = this.patterns.filter(p => p.id !== patternId);
    await this.saveUserPatterns();
    this.renderPatterns();
    this.showStatus('Pattern deleted', 'success');
  }
  
  async savePattern() {
    const name = document.getElementById('patternName').value.trim();
    const regex = document.getElementById('patternRegex').value.trim();
    const baseUrl = document.getElementById('patternUrl').value.trim();
    const description = document.getElementById('patternDescription').value.trim();
    const enabled = document.getElementById('patternEnabled').checked;
    
    if (!name || !regex || !baseUrl) {
      this.showStatus('Name, regex, and base URL are required', 'error');
      return;
    }
    
    const pattern = {
      id: this.currentEditingPattern?.id || Date.now().toString(),
      name,
      regex,
      baseUrl,
      description,
      enabled,
      isPredefined: false
    };
    
    // Validate pattern
    const validation = window.validatePattern ? window.validatePattern(pattern) : { isValid: true };
    if (!validation.isValid) {
      this.showStatus('Pattern validation failed: ' + validation.errors.join(', '), 'error');
      return;
    }
    
    if (this.currentEditingPattern) {
      // Update existing pattern
      const index = this.patterns.findIndex(p => p.id === pattern.id);
      if (index >= 0) {
        this.patterns[index] = pattern;
      }
    } else {
      // Add new pattern
      this.patterns.push(pattern);
    }
    
    await this.saveUserPatterns();
    this.hidePatternModal();
    this.renderPatterns();
    this.updateTestPatternDropdown();
    this.showStatus('Pattern saved successfully', 'success');
  }
  
  async saveUserPatterns() {
    const userPatterns = this.patterns.filter(p => !p.isPredefined);
    
    try {
      const success = await window.savePatternsChunked(userPatterns);
      if (!success) {
        throw new Error('Failed to save patterns using chunked storage');
      }
    } catch (error) {
      console.error('Failed to save patterns:', error);
      this.showStatus(`Failed to save patterns: ${error.message}`, 'error');
    }
  }
  
  validateCurrentPattern() {
    const regex = document.getElementById('patternRegex').value.trim();
    const baseUrl = document.getElementById('patternUrl').value.trim();
    const validationEl = document.getElementById('patternValidation');
    
    // Update real-time validation indicators
    this.updateValidationIndicator('regex', regex);
    this.updateValidationIndicator('url', baseUrl);
    
    if (!regex || !baseUrl) {
      validationEl.style.display = 'none';
      return;
    }
    
    const regexValidation = window.validateRegex ? window.validateRegex(regex) : { isValid: true };
    const urlValidation = window.validateBaseUrl ? window.validateBaseUrl(baseUrl, regexValidation.groupCount) : { isValid: true };
    
    validationEl.innerHTML = '';
    validationEl.className = 'pattern-validation show';
    
    if (regexValidation.isValid && urlValidation.isValid) {
      validationEl.classList.add('valid');
      validationEl.innerHTML = '<div class="validation-item">✓ Pattern is valid</div>';
      if (regexValidation.groupCount) {
        validationEl.innerHTML += `<div class="validation-item">✓ Found ${regexValidation.groupCount} capturing groups</div>`;
      }
    } else {
      validationEl.classList.add('invalid');
      const errors = [...(regexValidation.errors || []), ...(urlValidation.errors || [])];
      errors.forEach(error => {
        validationEl.innerHTML += `<div class="validation-item">✗ ${error}</div>`;
      });
    }
    
    // Update pattern testing when regex, url, or sample text changes
    this.updatePatternTesting();
  }
  
  updateValidationIndicator(type, value) {
    const iconEl = document.getElementById(`${type}ValidationIcon`);
    const messageEl = document.getElementById(`${type}ValidationMessage`);
    
    if (!iconEl || !messageEl) return;
    
    if (!value || value.trim() === '') {
      iconEl.className = 'validation-icon';
      messageEl.textContent = '';
      messageEl.className = 'validation-message';
      return;
    }
    
    let validation;
    if (type === 'regex') {
      validation = window.validateRegex ? window.validateRegex(value) : { isValid: true, errors: [] };
    } else if (type === 'url') {
      const regex = document.getElementById('patternRegex').value.trim();
      const regexValidation = window.validateRegex ? window.validateRegex(regex) : { groupCount: 0 };
      validation = window.validateBaseUrl ? window.validateBaseUrl(value, regexValidation.groupCount) : { isValid: true, errors: [] };
    }
    
    if (validation.isValid) {
      iconEl.className = 'validation-icon valid';
      messageEl.className = 'validation-message success';
      if (type === 'regex' && validation.groupCount) {
        messageEl.textContent = `Valid (${validation.groupCount} groups)`;
      } else {
        messageEl.textContent = 'Valid';
      }
    } else {
      iconEl.className = 'validation-icon invalid';
      messageEl.className = 'validation-message error';
      messageEl.textContent = validation.errors[0] || 'Invalid';
    }
  }
  
  updatePatternTesting() {
    const regex = document.getElementById('patternRegex').value.trim();
    const baseUrl = document.getElementById('patternUrl').value.trim();
    const sampleText = document.getElementById('sampleText').value.trim();
    const matchesEl = document.getElementById('matchesPreview');
    const urlsEl = document.getElementById('urlPreview');
    
    // Clear previous results
    matchesEl.innerHTML = '';
    urlsEl.innerHTML = '';
    
    if (!regex || !sampleText) {
      return;
    }
    
    // Test the pattern against sample text
    const testResult = window.testPattern ? window.testPattern(regex, sampleText) : { success: false };
    
    if (!testResult.success) {
      matchesEl.innerHTML = `<div style="color: #dc3545; font-style: italic;">Error testing pattern: ${testResult.error || 'Invalid regex'}</div>`;
      return;
    }
    
    if (testResult.matches.length === 0) {
      matchesEl.innerHTML = '<div style="color: #6c757d; font-style: italic;">No matches found in sample text</div>';
      return;
    }
    
    // Display matches
    testResult.matches.forEach((match, index) => {
      const matchEl = document.createElement('div');
      matchEl.className = 'match-item';
      
      const textEl = document.createElement('div');
      textEl.className = 'match-text';
      textEl.textContent = match.fullMatch;
      
      const groupsEl = document.createElement('div');
      groupsEl.className = 'match-groups';
      groupsEl.textContent = `Groups: [${match.groups.join(', ')}]`;
      
      matchEl.appendChild(textEl);
      matchEl.appendChild(groupsEl);
      matchesEl.appendChild(matchEl);
      
      // Generate URL if base URL is provided
      if (baseUrl) {
        const generatedUrl = window.generateUrl ? window.generateUrl(baseUrl, match.groups) : null;
        console.log('URL Generation Debug:', {
          baseUrl,
          groups: match.groups,
          generatedUrl
        });
        
        if (generatedUrl) {
          const urlEl = document.createElement('div');
          urlEl.className = 'generated-url';
          
          // Show the substitution details
          const substitutionInfo = match.groups.map((group, idx) => {
            return `{${idx}} → "${group}"`;
          }).join(', ');
          
          urlEl.innerHTML = `
            <div style="font-size: 11px; color: #666; margin-bottom: 4px;">
              Match: "${match.fullMatch}"
            </div>
            <div style="font-weight: bold; margin-bottom: 4px;">
              Generated URL: <a href="${generatedUrl}" target="_blank">${generatedUrl}</a>
            </div>
          `;
          urlsEl.appendChild(urlEl);
        } else {
          // Show error if URL generation failed
          const errorEl = document.createElement('div');
          errorEl.className = 'generated-url';
          errorEl.style.color = '#dc3545';
          errorEl.innerHTML = `
            <div>Error generating URL for match: "${match.fullMatch}"</div>
            <div style="font-size: 11px;">Template: ${baseUrl}</div>
            <div style="font-size: 11px;">Groups: [${match.groups.join(', ')}]</div>
          `;
          urlsEl.appendChild(errorEl);
        }
      }
    });
  }
  
  updateTestPatternDropdown() {
    const select = document.getElementById('testPattern');
    select.innerHTML = '<option value="">Select a pattern...</option>';
    
    this.patterns.forEach(pattern => {
      const option = document.createElement('option');
      option.value = pattern.id;
      option.textContent = pattern.name;
      select.appendChild(option);
    });
  }
  
  testSelectedPattern() {
    const patternId = document.getElementById('testPattern').value;
    const testText = document.getElementById('testText').value;
    const resultsEl = document.getElementById('testResults');
    
    if (!patternId) {
      this.showStatus('Please select a pattern to test', 'warning');
      return;
    }
    
    const pattern = this.patterns.find(p => p.id === patternId);
    if (!pattern) return;
    
    const testResult = window.testPattern ? window.testPattern(pattern.regex, testText) : { success: false };
    
    resultsEl.innerHTML = '';
    resultsEl.classList.add('show');
    
    if (testResult.success && testResult.matches.length > 0) {
      resultsEl.innerHTML = `<h4>Found ${testResult.matches.length} matches:</h4>`;
      
      testResult.matches.forEach(match => {
        const url = window.generateUrl ? window.generateUrl(pattern.baseUrl, match.groups) : '#';
        const matchEl = document.createElement('div');
        matchEl.className = 'match-item';
        matchEl.innerHTML = `
          <div class="match-text">${this.escapeHtml(match.fullMatch)}</div>
          <div class="match-url">→ ${this.escapeHtml(url)}</div>
        `;
        resultsEl.appendChild(matchEl);
      });
    } else {
      resultsEl.innerHTML = '<div class="no-matches">No matches found for this pattern</div>';
    }
  }
  
  previewAllPatterns() {
    const testText = document.getElementById('testText').value;
    const resultsEl = document.getElementById('testResults');
    
    resultsEl.innerHTML = '';
    resultsEl.classList.add('show');
    
    let totalMatches = 0;
    const enabledPatterns = this.patterns.filter(p => p.enabled !== false);
    
    enabledPatterns.forEach(pattern => {
      const testResult = window.testPattern ? window.testPattern(pattern.regex, testText) : { success: false };
      
      if (testResult.success && testResult.matches.length > 0) {
        totalMatches += testResult.matches.length;
        
        const sectionEl = document.createElement('div');
        sectionEl.innerHTML = `<h4>${this.escapeHtml(pattern.name)} (${testResult.matches.length} matches):</h4>`;
        resultsEl.appendChild(sectionEl);
        
        testResult.matches.forEach(match => {
          const url = window.generateUrl ? window.generateUrl(pattern.baseUrl, match.groups) : '#';
          const matchEl = document.createElement('div');
          matchEl.className = 'match-item';
          matchEl.innerHTML = `
            <div class="match-text">${this.escapeHtml(match.fullMatch)}</div>
            <div class="match-url">→ ${this.escapeHtml(url)}</div>
          `;
          resultsEl.appendChild(matchEl);
        });
      }
    });
    
    if (totalMatches === 0) {
      resultsEl.innerHTML = '<div class="no-matches">No matches found for any enabled patterns</div>';
    } else {
      resultsEl.insertAdjacentHTML('afterbegin', `<h4>Total: ${totalMatches} matches from ${enabledPatterns.length} patterns</h4><hr>`);
    }
  }
  
  updateLivePreview() {
    const testText = document.getElementById('testText').value;
    const previewEl = document.getElementById('livePreview');
    
    if (!testText.trim()) {
      previewEl.innerHTML = '<em>Enter text above to see live preview...</em>';
      return;
    }
    
    // Simulate the text processing
    let processedText = testText;
    const enabledPatterns = this.patterns.filter(p => p.enabled !== false);
    
    enabledPatterns.forEach(pattern => {
      try {
        const regex = new RegExp(pattern.regex, 'gi');
        processedText = processedText.replace(regex, (match, ...groups) => {
          const url = window.generateUrl ? window.generateUrl(pattern.baseUrl, groups.slice(0, -2)) : '#';
          return `<a href="${url}" class="cite-linker-pro-link cite-linker-pro-dashed" target="_blank" title="${pattern.name}: ${url}">${match}</a>`;
        });
      } catch (error) {
        console.warn('Regex error in pattern:', pattern.name, error);
      }
    });
    
    previewEl.innerHTML = processedText;
  }
  
  exportConfig() {
    try {
      const xmlContent = window.exportToXML ? window.exportToXML(this.patterns, this.settings) : '';
      if (xmlContent) {
        window.downloadXML(xmlContent, 'cite-linker-config.xml');
        this.showStatus('Configuration exported successfully', 'success');
      } else {
        throw new Error('Export function not available');
      }
    } catch (error) {
      console.error('Export failed:', error);
      this.showStatus('Failed to export configuration', 'error');
    }
  }
  
  triggerImport() {
    document.getElementById('importFile').click();
  }
  
  async importConfig(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    try {
      const xmlContent = await window.readXMLFile(file);
      const importResult = window.importFromXMLAny ? window.importFromXMLAny(xmlContent) : { success: false };
      
      if (!importResult.success) {
        throw new Error(importResult.error || 'Import failed');
      }
      
      // Confirm import
      const data = importResult.data;
      const formatInfo = data.metadata?.format ? ` (${data.metadata.format} format)` : '';
      const message = `Import ${data.patterns.length} patterns and ${Object.keys(data.settings).length} settings${formatInfo}?`;
      
      if (!confirm(message)) {
        return;
      }
      
      // Apply imported settings
      Object.assign(this.settings, data.settings);
      
      // Replace user patterns with imported ones (keep predefined)
      const importedUserPatterns = data.patterns.filter(p => !p.isPredefined);
      this.patterns = [...this.predefinedPatterns, ...importedUserPatterns];
      
      // Save to storage
      await chrome.storage.sync.set(this.settings);
      await this.saveUserPatterns();
      
      // Update UI
      this.updateUI();
      this.renderPatterns();
      this.updateTestPatternDropdown();
      
      this.showStatus(`Successfully imported ${data.patterns.length} patterns${formatInfo}`, 'success');
      
    } catch (error) {
      console.error('Import failed:', error);
      this.showStatus('Import failed: ' + error.message, 'error');
    }
    
    // Clear file input
    event.target.value = '';
  }
  
  showStatus(message, type = 'success') {
    const statusEl = document.getElementById('status');
    statusEl.textContent = message;
    statusEl.className = `status ${type}`;
    statusEl.style.display = 'block';
    
    setTimeout(() => {
      statusEl.style.display = 'none';
    }, 5000);
  }
  
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  handleDragStart(e) {
    e.dataTransfer.setData('text/plain', e.currentTarget.dataset.patternId);
    e.currentTarget.classList.add('dragging');
  }

  handleDragOver(e) {
    e.preventDefault();
    const draggingElement = document.querySelector('.dragging');
    const afterElement = this.getDragAfterElement(e.currentTarget.parentElement, e.clientY);
    
    if (afterElement == null) {
      e.currentTarget.parentElement.appendChild(draggingElement);
    } else {
      e.currentTarget.parentElement.insertBefore(draggingElement, afterElement);
    }
  }

  handleDrop(e) {
    e.preventDefault();
    this.updatePatternOrder();
  }

  handleDragEnd(e) {
    e.currentTarget.classList.remove('dragging');
  }

  getDragAfterElement(container, y) {
    const draggableElements = [...container.querySelectorAll('.pattern-item:not(.dragging)')];
    
    return draggableElements.reduce((closest, child) => {
      const box = child.getBoundingClientRect();
      const offset = y - box.top - box.height / 2;
      
      if (offset < 0 && offset > closest.offset) {
        return { offset: offset, element: child };
      } else {
        return closest;
      }
    }, { offset: Number.NEGATIVE_INFINITY }).element;
  }

  async updatePatternOrder() {
    const container = document.getElementById('patternsList');
    const patternElements = container.querySelectorAll('.pattern-item');
    const orderedPatternIds = Array.from(patternElements).map(el => el.dataset.patternId);
    
    // Reorder patterns array based on new order
    const reorderedPatterns = [];
    orderedPatternIds.forEach(id => {
      const pattern = this.patterns.find(p => p.id === id);
      if (pattern) {
        reorderedPatterns.push(pattern);
      }
    });
    
    this.patterns = reorderedPatterns;
    
    // Save the new order
    await this.saveUserPatterns();
    
    // Re-render to update numbering
    this.renderPatterns();
    
    this.showStatus('Pattern order updated', 'success');
  }
}

// Initialize options manager when DOM is ready
let optionsManager;
document.addEventListener('DOMContentLoaded', () => {
  optionsManager = new OptionsManager();
  // Make it globally accessible for onclick handlers
  window.optionsManager = optionsManager;
});
