/**
 * Content script for Cite Linker Pro
 * Handles text processing and link injection on web pages
 */


class CiteLinkManager {
  constructor() {
    this.processor = null;
    this.isEnabled = true;
    this.settings = {};
    this.observer = null;
    this.processingQueue = [];
    this.isProcessing = false;
    this.processedNodes = new WeakSet();
    this.tooltipElement = null;
    
    this.init();
  }
  
  async init() {
    try {
      await this.loadSettings();
      await this.checkDomainPermissions();
      
      if (this.isEnabled) {
        await this.setupTextProcessor();
        this.setupTooltipSystem();
        this.setupMessageListener();
        
        // Only auto-process if auto-scan is enabled
        if (this.autoScan) {
          this.setupMutationObserver();
          this.processExistingContent();
        }
      }
    } catch (error) {
      console.error('CiteLinkManager initialization failed:', error);
    }
  }
  
  async loadSettings() {
    try {
      this.settings = await window.getSettings();
      this.isEnabled = this.settings[window.STORAGE_KEYS.ENABLED];
      this.autoScan = this.settings.autoScan === true; // Default to false, only true if explicitly set
    } catch (error) {
      console.error('Failed to load settings:', error);
      this.isEnabled = false;
      this.autoScan = false;
    }
  }
  
  async checkDomainPermissions() {
    if (!this.isEnabled) return;
    
    const currentDomain = window.location.hostname.toLowerCase();
    const domainMode = this.settings[window.STORAGE_KEYS.DOMAIN_MODE];
    const domains = this.settings[window.STORAGE_KEYS.DOMAINS] || [];
    
    switch (domainMode) {
      case 'whitelist':
        this.isEnabled = domains.some(domain => 
          currentDomain === domain || currentDomain.endsWith('.' + domain)
        );
        break;
        
      case 'blacklist':
        this.isEnabled = !domains.some(domain => 
          currentDomain === domain || currentDomain.endsWith('.' + domain)
        );
        break;
        
      case 'all':
      default:
        // Already enabled, no additional checks needed
        break;
    }
  }
  
  async setupTextProcessor() {
    try {
      // Initialize predefined patterns
      if (window.initializePredefinedPatterns) {
        await window.initializePredefinedPatterns();
      }
      
      const userPatterns = await window.loadPatternsChunked();
      const predefinedPatterns = window.getPredefinedPatterns ? 
        (await window.getPredefinedPatterns()).filter(p => p.enabled !== false) : 
        [];
      const allPatterns = [...predefinedPatterns, ...userPatterns];
      
      this.processor = new window.TextProcessor(
        allPatterns,
        this.settings[window.STORAGE_KEYS.LINK_STYLE] || 'dashed',
        this.settings[window.STORAGE_KEYS.SHOW_TOOLTIPS] !== false
      );
    } catch (error) {
      console.error('Failed to setup text processor:', error);
      // Fallback to user patterns only
      const userPatterns = await window.loadPatternsChunked();
      this.processor = new window.TextProcessor(
        userPatterns,
        this.settings[window.STORAGE_KEYS.LINK_STYLE] || 'dashed',
        this.settings[window.STORAGE_KEYS.SHOW_TOOLTIPS] !== false
      );
    }
  }
  
  setupMutationObserver() {
    this.observer = new MutationObserver((mutations) => {
      this.queueMutations(mutations);
    });
    
    this.observer.observe(document.body, {
      childList: true,
      subtree: true,
      characterData: true
    });
  }
  
  setupTooltipSystem() {
    if (!this.settings[window.STORAGE_KEYS.SHOW_TOOLTIPS]) return;
    
    this.tooltipElement = document.createElement('div');
    this.tooltipElement.className = 'cite-linker-pro-tooltip';
    document.body.appendChild(this.tooltipElement);
    
    // Add event listeners for tooltip
    document.addEventListener('mouseover', this.handleTooltipShow.bind(this));
    document.addEventListener('mouseout', this.handleTooltipHide.bind(this));
  }
  
  setupMessageListener() {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      switch (message.type) {
        case 'TOGGLE_EXTENSION':
          this.toggleExtension();
          sendResponse({ success: true });
          break;
          
        case 'SETTINGS_UPDATED':
          this.handleSettingsUpdate();
          sendResponse({ success: true });
          break;
          
        case 'PROCESS_PAGE':
          this.processExistingContent();
          sendResponse({ success: true });
          break;
          
        case 'PROCESS_SELECTION':
          this.processSelectedText();
          sendResponse({ success: true });
          break;
          
        case 'UPDATE_AUTO_SCAN':
          this.handleAutoScanUpdate(message.autoScan);
          sendResponse({ success: true });
          break;
          
        case 'GET_STATS':
          const stats = this.getStats();
          sendResponse({ success: true, stats });
          break;
          
        default:
          sendResponse({ success: false, error: 'Unknown message type' });
      }
    });
  }
  
  queueMutations(mutations) {
    const nodesToProcess = [];
    
    mutations.forEach(mutation => {
      if (mutation.type === 'childList') {
        mutation.addedNodes.forEach(node => {
          if (node.nodeType === Node.TEXT_NODE) {
            nodesToProcess.push(node);
          } else if (node.nodeType === Node.ELEMENT_NODE) {
            const textNodes = this.getTextNodes(node);
            nodesToProcess.push(...textNodes);
          }
        });
      } else if (mutation.type === 'characterData') {
        nodesToProcess.push(mutation.target);
      }
    });
    
    if (nodesToProcess.length > 0) {
      this.queueTextProcessing(nodesToProcess);
    }
  }
  
  queueTextProcessing(nodes) {
    this.processingQueue.push(...nodes);
    
    if (!this.isProcessing) {
      this.processQueue();
    }
  }
  
  async processQueue() {
    this.isProcessing = true;
    
    const batchSize = 10; // Process in small batches
    
    while (this.processingQueue.length > 0) {
      const batch = this.processingQueue.splice(0, batchSize);
      
      // Use requestIdleCallback for performance
      await new Promise(resolve => {
        const processNextBatch = (deadline) => {
          let processed = 0;
          
          while (processed < batch.length && deadline.timeRemaining() > 0) {
            const node = batch[processed];
            
            if (!this.processedNodes.has(node) && 
                node.isConnected && 
                this.processor) {
              
              try {
                if (this.processor.processTextNode(node)) {
                  this.processedNodes.add(node);
                }
              } catch (error) {
                console.warn('Error processing text node:', error);
              }
            }
            
            processed++;
          }
          
          resolve();
        };
        
        if (window.requestIdleCallback) {
          requestIdleCallback(processNextBatch, { timeout: 100 });
        } else {
          setTimeout(() => processNextBatch({ timeRemaining: () => 5 }), 0);
        }
      });
    }
    
    this.isProcessing = false;
  }
  
  processExistingContent() {
    if (!this.processor) return;
    
    const textNodes = this.getTextNodes(document.body);
    this.queueTextProcessing(textNodes);
  }
  
  getTextNodes(element) {
    const textNodes = [];
    const walker = document.createTreeWalker(
      element,
      NodeFilter.SHOW_TEXT,
      {
        acceptNode: (node) => {
          // Skip if parent is script, style, or already processed
          const parent = node.parentElement;
          if (!parent) return NodeFilter.FILTER_REJECT;
          
          const tagName = parent.tagName?.toLowerCase();
          if (['script', 'style', 'noscript'].includes(tagName)) {
            return NodeFilter.FILTER_REJECT;
          }
          
          // Skip if inside existing link or citation
          if (parent.closest('a, [data-cite-linker-pro]')) {
            return NodeFilter.FILTER_REJECT;
          }
          
          // Skip empty or whitespace-only nodes
          if (!node.textContent.trim()) {
            return NodeFilter.FILTER_REJECT;
          }
          
          return NodeFilter.FILTER_ACCEPT;
        }
      }
    );
    
    let node;
    while (node = walker.nextNode()) {
      textNodes.push(node);
    }
    
    return textNodes;
  }
  
  handleTooltipShow(event) {
    const target = event.target;
    
    if (target.hasAttribute('data-cite-linker-pro') && 
        target.hasAttribute('data-cite-tooltip')) {
      
      const tooltipText = target.getAttribute('data-cite-tooltip');
      this.showTooltip(event, tooltipText);
    }
  }
  
  handleTooltipHide(event) {
    if (!event.relatedTarget || 
        !event.relatedTarget.hasAttribute('data-cite-linker-pro')) {
      this.hideTooltip();
    }
  }
  
  showTooltip(event, text) {
    if (!this.tooltipElement) return;
    
    this.tooltipElement.textContent = text;
    this.tooltipElement.classList.add('show');
    
    // Position tooltip
    const rect = event.target.getBoundingClientRect();
    const tooltipRect = this.tooltipElement.getBoundingClientRect();
    
    let left = rect.left + (rect.width / 2) - (tooltipRect.width / 2);
    let top = rect.top - tooltipRect.height - 10;
    
    // Adjust if tooltip goes off screen
    if (left < 10) left = 10;
    if (left + tooltipRect.width > window.innerWidth - 10) {
      left = window.innerWidth - tooltipRect.width - 10;
    }
    if (top < 10) {
      top = rect.bottom + 10;
    }
    
    this.tooltipElement.style.left = left + window.scrollX + 'px';
    this.tooltipElement.style.top = top + window.scrollY + 'px';
  }
  
  hideTooltip() {
    if (this.tooltipElement) {
      this.tooltipElement.classList.remove('show');
    }
  }
  
  async toggleExtension() {
    this.isEnabled = !this.isEnabled;
    
    if (this.isEnabled) {
      await this.loadSettings();
      await this.setupTextProcessor();
      this.processExistingContent();
    } else {
      this.removeAllCitations();
    }
  }
  
  async handleSettingsUpdate() {
    await this.loadSettings();
    await this.checkDomainPermissions();
    
    if (this.isEnabled) {
      await this.setupTextProcessor();
      this.removeAllCitations();
      this.processExistingContent();
    } else {
      this.removeAllCitations();
    }
  }
  
  processSelectedText() {
    const selection = window.getSelection();
    if (!selection.rangeCount) return;
    
    const range = selection.getRangeAt(0);
    const selectedText = range.toString().trim();
    
    if (!selectedText || !this.processor) return;
    
    // Create temporary text node to process
    const tempNode = document.createTextNode(selectedText);
    const tempContainer = document.createElement('div');
    tempContainer.appendChild(tempNode);
    
    // Process the text
    if (this.processor.processTextNode(tempNode)) {
      // Replace selection with processed content
      try {
        range.deleteContents();
        range.insertNode(tempContainer.firstChild);
        selection.removeAllRanges();
      } catch (error) {
        console.warn('Failed to replace selected text:', error);
      }
    }
  }
  
  handleAutoScanUpdate(autoScan) {
    this.autoScan = autoScan;
    
    if (autoScan) {
      // Start auto-scanning: setup mutation observer and process existing content
      this.setupMutationObserver();
      this.processExistingContent();
    } else {
      // Stop auto-scanning: disconnect mutation observer
      if (this.observer) {
        this.observer.disconnect();
        this.observer = null;
      }
    }
  }
  
  getStats() {
    const citationLinks = document.querySelectorAll('.cite-linker-pro-link');
    return {
      citations: citationLinks.length,
      patterns: this.processor ? this.processor.patterns.length : 0
    };
  }

  removeAllCitations() {
    const citations = document.querySelectorAll('[data-cite-linker-pro]');
    citations.forEach(citation => {
      const originalText = citation.getAttribute('data-original-text');
      if (originalText) {
        const textNode = document.createTextNode(originalText);
        citation.parentNode?.replaceChild(textNode, citation);
      }
    });
    
    this.processedNodes = new WeakSet();
  }
  
  destroy() {
    if (this.observer) {
      this.observer.disconnect();
    }
    
    if (this.tooltipElement) {
      this.tooltipElement.remove();
    }
    
    this.removeAllCitations();
    
    document.removeEventListener('mouseover', this.handleTooltipShow);
    document.removeEventListener('mouseout', this.handleTooltipHide);
  }
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    new CiteLinkManager();
  });
} else {
  new CiteLinkManager();
}